<?php
/**
 * Post Type Functions
 *
 * @package     MVPWP CPT Plugin
 * @subpackage  Functions
 * @copyright   Copyright (c) 2016, Brad Williams
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Registers and sets up the Projects custom post type
 *
 * @since 1.0
 * @return void
 */
function mvpwp_setup_mvpwp_post_types() {

  $archives = defined( 'MVPWP_DISABLE_ARCHIVE' ) && MVPWP_DISABLE_ARCHIVE ? false : true;
  $slug     = defined( 'MVPWP_SLUG' ) ? MVPWP_SLUG : 'projects';
  $rewrite  = defined( 'MVPWP_DISABLE_REWRITE' ) && MVPWP_DISABLE_REWRITE ? false : array('slug' => $slug, 'with_front' => false);

  $project_labels =  apply_filters( 'mvpwp_project_labels', array(
    'name'                  => _x( '%2$s', 'project post type name', 'mvpwp' ),
    'singular_name'         => _x( '%1$s', 'singular project post type name', 'mvpwp' ),
    'add_new'               => __( 'Add New', 'mvpwp' ),
    'add_new_item'          => __( 'Add New %1$s', 'mvpwp' ),
    'edit_item'             => __( 'Edit %1$s', 'mvpwp' ),
    'new_item'              => __( 'New %1$s', 'mvpwp' ),
    'all_items'             => __( 'All %2$s', 'mvpwp' ),
    'view_item'             => __( 'View %1$s', 'mvpwp' ),
    'search_items'          => __( 'Search %2$s', 'mvpwp' ),
    'not_found'             => __( 'No %2$s found', 'mvpwp' ),
    'not_found_in_trash'    => __( 'No %2$s found in Trash', 'mvpwp' ),
    'parent_item_colon'     => '',
    'menu_name'             => _x( '%2$s', 'project post type menu name', 'mvpwp' ),
    'featured_image'        => __( '%1$s Image', 'mvpwp' ),
    'set_featured_image'    => __( 'Set %1$s Image', 'mvpwp' ),
    'remove_featured_image' => __( 'Remove %1$s Image', 'mvpwp' ),
    'use_featured_image'    => __( 'Use as %1$s Image', 'mvpwp' ),
    'filter_items_list'     => __( 'Filter %2$s list', 'mvpwp' ),
    'items_list_navigation' => __( '%2$s list navigation', 'mvpwp' ),
    'items_list'            => __( '%2$s list', 'mvpwp' ),
  ) );

  foreach ( $project_labels as $key => $value ) {
    $project_labels[ $key ] = sprintf( $value, mvpwp_get_label_singular(), mvpwp_get_label_plural() );
  }

  $project_args = array(
    'labels'             => $project_labels,
    'public'             => true,
    'publicly_queryable' => true,
    'show_ui'            => true,
    'show_in_menu'       => true,
    'query_var'          => true,
    'rewrite'            => $rewrite,
    'map_meta_cap'       => true,
    'has_archive'        => $archives,
    'hierarchical'       => false,
    'supports'           => apply_filters( 'mvpwp_project_supports', array( 'title', 'editor', 'thumbnail', 'excerpt', 'revisions', 'author' ) ),
  );
  register_post_type( 'project', apply_filters( 'mvpwp_project_post_type_args', $project_args ) );

}
add_action( 'init', 'mvpwp_setup_mvpwp_post_types', 1 );

/**
 * Get Default Labels
 *
 * @since 1.0.8.3
 * @return array $defaults Default labels
 */
function mvpwp_get_default_labels() {
  $defaults = array(
     'singular' => __( 'Project', 'mvpwp' ),
     'plural'   => __( 'Projects','mvpwp' )
  );
  return apply_filters( 'mvpwp_default_projects_name', $defaults );
}

/**
 * Get Singular Label
 *
 * @since 1.0.8.3
 *
 * @param bool $lowercase
 * @return string $defaults['singular'] Singular label
 */
function mvpwp_get_label_singular( $lowercase = false ) {
  $defaults = mvpwp_get_default_labels();
  return ($lowercase) ? strtolower( $defaults['singular'] ) : $defaults['singular'];
}

/**
 * Get Plural Label
 *
 * @since 1.0.8.3
 * @return string $defaults['plural'] Plural label
 */
function mvpwp_get_label_plural( $lowercase = false ) {
  $defaults = mvpwp_get_default_labels();
  return ( $lowercase ) ? strtolower( $defaults['plural'] ) : $defaults['plural'];
}

/**
 * Change default "Enter title here" input
 *
 * @since 1.4.0.2
 * @param string $title Default title placeholder text
 * @return string $title New placeholder text
 */
function mvpwp_change_default_title( $title ) {
   // If a frontend plugin uses this filter (check extensions before changing this function)
   if ( !is_admin() ) {
    $label = mvpwp_get_label_singular();
    $title = sprintf( __( 'Enter %s name here', 'mvpwp' ), $label );
    return $title;
   }

   $screen = get_current_screen();

   if ( 'project' == $screen->post_type ) {
    $label = mvpwp_get_label_singular();
    $title = sprintf( __( 'Enter %s name here', 'mvpwp' ), $label );
   }

   return $title;
}
add_filter( 'enter_title_here', 'mvpwp_change_default_title' );

/**
 * Registers the custom taxonomies for the projects custom post type
 *
 * @since 1.0
 * @return void
*/
function mvpwp_setup_project_taxonomies() {

  $slug     = defined( 'MVPWP_SLUG' ) ? MVPWP_SLUG : 'projects';

  /** Categories */
  $category_labels = array(
    'name'              => sprintf( _x( '%s Categories', 'taxonomy general name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'singular_name'     => sprintf( _x( '%s Category', 'taxonomy singular name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'search_items'      => sprintf( __( 'Search %s Categories', 'mvpwp' ), mvpwp_get_label_singular() ),
    'all_items'         => sprintf( __( 'All %s Categories', 'mvpwp' ), mvpwp_get_label_singular() ),
    'parent_item'       => sprintf( __( 'Parent %s Category', 'mvpwp' ), mvpwp_get_label_singular() ),
    'parent_item_colon' => sprintf( __( 'Parent %s Category:', 'mvpwp' ), mvpwp_get_label_singular() ),
    'edit_item'         => sprintf( __( 'Edit %s Category', 'mvpwp' ), mvpwp_get_label_singular() ),
    'update_item'       => sprintf( __( 'Update %s Category', 'mvpwp' ), mvpwp_get_label_singular() ),
    'add_new_item'      => sprintf( __( 'Add New %s Category', 'mvpwp' ), mvpwp_get_label_singular() ),
    'new_item_name'     => sprintf( __( 'New %s Category Name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'menu_name'         => __( 'Categories', 'mvpwp' ),
  );

  $category_args = apply_filters( 'mvpwp_project_category_args', array(
      'hierarchical' => true,
      'labels'       => apply_filters('mvpwp_project_category_labels', $category_labels),
      'show_ui'      => true,
      'query_var'    => 'project_category',
      'rewrite'      => array('slug' => $slug . '/category', 'with_front' => false, 'hierarchical' => true )
    )
  );
  register_taxonomy( 'project_category', array('project'), $category_args );
  register_taxonomy_for_object_type( 'project_category', 'project' );

  /** Tags */
  $tag_labels = array(
    'name'                  => sprintf( _x( '%s Tags', 'taxonomy general name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'singular_name'         => sprintf( _x( '%s Tag', 'taxonomy singular name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'search_items'          => sprintf( __( 'Search %s Tags', 'mvpwp' ), mvpwp_get_label_singular() ),
    'all_items'             => sprintf( __( 'All %s Tags', 'mvpwp' ), mvpwp_get_label_singular() ),
    'parent_item'           => sprintf( __( 'Parent %s Tag', 'mvpwp' ), mvpwp_get_label_singular() ),
    'parent_item_colon'     => sprintf( __( 'Parent %s Tag:', 'mvpwp' ), mvpwp_get_label_singular() ),
    'edit_item'             => sprintf( __( 'Edit %s Tag', 'mvpwp' ), mvpwp_get_label_singular() ),
    'update_item'           => sprintf( __( 'Update %s Tag', 'mvpwp' ), mvpwp_get_label_singular() ),
    'add_new_item'          => sprintf( __( 'Add New %s Tag', 'mvpwp' ), mvpwp_get_label_singular() ),
    'new_item_name'         => sprintf( __( 'New %s Tag Name', 'mvpwp' ), mvpwp_get_label_singular() ),
    'menu_name'             => __( 'Tags', 'mvpwp' ),
    'choose_from_most_used' => sprintf( __( 'Choose from most used %s tags', 'mvpwp' ), mvpwp_get_label_singular() ),
  );

  $tag_args = apply_filters( 'mvpwp_project_tag_args', array(
      'hierarchical' => false,
      'labels'       => apply_filters( 'mvpwp_project_tag_labels', $tag_labels ),
      'show_ui'      => true,
      'query_var'    => 'project_tag',
      'rewrite'      => array( 'slug' => $slug . '/tag', 'with_front' => false, 'hierarchical' => true  )
    )
  );
  register_taxonomy( 'project_tag', array( 'project' ), $tag_args );
  register_taxonomy_for_object_type( 'project_tag', 'project' );
}
add_action( 'init', 'mvpwp_setup_project_taxonomies', 0 );

/**
 * Get the singular and plural labels for a project taxonomy
 *
 * @since  2.4
 * @param  string $taxonomy The Taxonomy to get labels for
 * @return array            Associative array of labels (name = plural)
 */
function mvpwp_get_taxonomy_labels( $taxonomy = 'project_category' ) {

  $allowed_taxonomies = apply_filters( 'mvpwp_allowed_project_taxonomies', array( 'project_category', 'project_tag' ) );

  if ( ! in_array( $taxonomy, $allowed_taxonomies ) ) {
    return false;
  }

  $labels   = array();
  $taxonomy = get_taxonomy( $taxonomy );

  if ( false !== $taxonomy ) {
    $singular = $taxonomy->labels->singular_name;
    $name     = $taxonomy->labels->name;

    $labels = array(
      'name'          => $name,
      'singular_name' => $singular,
    );
  }

  return apply_filters( 'mvpwp_get_taxonomy_labels', $labels, $taxonomy );

}